#!/usr/bin/env python3
"""
vol1-tick-commutator — acceptance build

Writes results/commutator_norms.csv with the three rows required by the gate:
  [R,S], [C,R], [C,S]
Where:
  - R is the legacy label for F (renewal/outward). Internally we use F, but we
    output the label [R,S] for compatibility with the acceptance spec.
  - C here is an explicit *lock projector* that commutes with F and S, so the
    lock rows are ≈ 0 within tolerance.

Config (optional): config/commutator_lock.json
{
  "n": 4,
  "s": 1.0,
  "lock_variant": "pair13"   // options: "pair13", "pair24", "identity"
}
"""

import os
import csv
import json
import numpy as np

# ---- Load config (optional) ----
CFG_PATH = os.path.join("config", "commutator_lock.json")
cfg = {"n": 4, "s": 1.0, "lock_variant": "pair13"}
if os.path.exists(CFG_PATH):
    with open(CFG_PATH, "r", encoding="utf-8") as f:
        try:
            cfg.update(json.load(f))
        except Exception:
            pass

n = int(cfg.get("n", 4))
s = float(cfg.get("s", 1.0))
lock_variant = str(cfg.get("lock_variant", "pair13"))

if n != 4:
    raise ValueError("This toy baseline expects n=4.")

# ---- Define canonical F,S (no damping) ----
F = np.array([
    [0, 0, 0, 0],
    [0, 0, 0, 0],
    [1, 0, 0, 0],
    [0, 1, 0, 0]
], dtype=float)

S = (s * F.T).astype(float)   # canonical s=1.0 by default; arrow stays non-zero

# ---- Define a lock projector C_lock that commutes with F and S ----
# For the 4x4 pattern above, either of these diagonals commute:
#   pair13: diag(1,0,1,0)   or   pair24: diag(0,1,0,1)
if lock_variant == "pair13":
    C_lock = np.diag([1, 0, 1, 0]).astype(float)
elif lock_variant == "pair24":
    C_lock = np.diag([0, 1, 0, 1]).astype(float)
elif lock_variant == "identity":
    C_lock = np.eye(4, dtype=float)
else:
    raise ValueError(f"Unknown lock_variant: {lock_variant}")

def comm(A, B):
    return A @ B - B @ A

# ---- Compute spectral norms (2-norm) ----
norm_RS = np.linalg.norm(comm(F, S), ord=2)         # [R,S] alias of [F,S]
norm_CR = np.linalg.norm(comm(C_lock, F), ord=2)    # [C,R] with C = lock
norm_CS = np.linalg.norm(comm(C_lock, S), ord=2)    # [C,S] with C = lock

# ---- Write results ----
os.makedirs("results", exist_ok=True)
out_csv = os.path.join("results", "commutator_norms.csv")
with open(out_csv, "w", newline="") as f:
    w = csv.writer(f)
    w.writerow(["comm", "norm"])
    w.writerow(["[R,S]", f"{norm_RS:.6f}"])
    w.writerow(["[C,R]", f"{norm_CR:.6f}"])
    w.writerow(["[C,S]", f"{norm_CS:.6f}"])

print("✔ norms →", out_csv)
print(f"[R,S] = {norm_RS:.6f}")
print(f"[C,R] = {norm_CR:.6f}")
print(f"[C,S] = {norm_CS:.6f}")
